document.addEventListener('DOMContentLoaded', function() {
    // Mobil menü tüklama işlemi
    const mobileMenuToggle = document.querySelector('.mobile-menu-toggle');
    const navMenu = document.querySelector('.nav-menu');
    
    if (mobileMenuToggle) {
        mobileMenuToggle.addEventListener('click', function() {
            navMenu.classList.toggle('active');
        });
    }
    
    // Uyarı mesajları için otomatik kapanma
    const alerts = document.querySelectorAll('.alert');
    
    alerts.forEach(alert => {
        // 5 saniye sonra uyarıyı gizle
        setTimeout(() => {
            alert.style.opacity = '0';
            alert.style.transition = 'opacity 0.5s';
            
            // Tamamen kaldırmadan önce geçiş efektinin tamamlanmasını bekle
            setTimeout(() => {
                alert.style.display = 'none';
            }, 500);
        }, 5000);
        
        // Kapatma düğmesi için olay dinleyicisi
        const closeBtn = alert.querySelector('.close-alert');
        if (closeBtn) {
            closeBtn.addEventListener('click', () => {
                alert.style.opacity = '0';
                alert.style.transition = 'opacity 0.5s';
                
                setTimeout(() => {
                    alert.style.display = 'none';
                }, 500);
            });
        }
    });
    
    // Form doğrulama işlemleri
    const forms = document.querySelectorAll('form[data-validate="true"]');
    
    forms.forEach(form => {
        form.addEventListener('submit', function(event) {
            const requiredFields = form.querySelectorAll('[required]');
            let isValid = true;
            
            requiredFields.forEach(field => {
                if (!field.value.trim()) {
                    isValid = false;
                    field.classList.add('is-invalid');
                    
                    // Hata mesajı oluştur
                    const errorMsg = document.createElement('div');
                    errorMsg.className = 'invalid-feedback';
                    errorMsg.textContent = 'Bu alan zorunludur';
                    
                    // Eğer hata mesajı yoksa ekle
                    if (!field.nextElementSibling || !field.nextElementSibling.classList.contains('invalid-feedback')) {
                        field.parentNode.insertBefore(errorMsg, field.nextSibling);
                    }
                } else {
                    field.classList.remove('is-invalid');
                    
                    // Varsa hata mesajını kaldır
                    if (field.nextElementSibling && field.nextElementSibling.classList.contains('invalid-feedback')) {
                        field.nextElementSibling.remove();
                    }
                }
            });
            
            if (!isValid) {
                event.preventDefault();
            }
        });
    });
    
    // Silme işlemleri için onay kutusu
    const deleteBtns = document.querySelectorAll('.delete-btn');
    
    deleteBtns.forEach(btn => {
        btn.addEventListener('click', function(event) {
            if (!confirm('Bu öğeyi silmek istediğinizden emin misiniz?')) {
                event.preventDefault();
            }
        });
    });
    
    // Yazı içeriği önizleme
    const contentInput = document.getElementById('icerik');
    const previewBtn = document.getElementById('preview-btn');
    const previewContainer = document.getElementById('preview-container');
    
    if (contentInput && previewBtn && previewContainer) {
        previewBtn.addEventListener('click', function() {
            const content = contentInput.value;
            previewContainer.innerHTML = `<div class="yazi-icerik">${content}</div>`;
            previewContainer.style.display = 'block';
        });
    }
});

// AlertV2 - Gelişmiş uyarı mesajları için JavaScript kütüphanesi
const AlertV2 = {
    init: function() {
        // Stil elementini oluştur
        const style = document.createElement('style');
        style.textContent = `
            .alertv2-container {
                position: fixed;
                top: 20px;
                right: 20px;
                z-index: 9999;
                max-width: 350px;
            }
            
            .alertv2 {
                padding: 15px;
                border-radius: 5px;
                margin-bottom: 10px;
                box-shadow: 0 3px 10px rgba(0, 0, 0, 0.2);
                display: flex;
                align-items: flex-start;
                transform: translateX(120%);
                transition: transform 0.3s ease;
                background: white;
            }
            
            .alertv2.show {
                transform: translateX(0);
            }
            
            .alertv2-icon {
                margin-right: 15px;
                font-size: 24px;
            }
            
            .alertv2-success {
                border-left: 5px solid var(--success-color);
            }
            
            .alertv2-success .alertv2-icon {
                color: var(--success-color);
            }
            
            .alertv2-error {
                border-left: 5px solid var(--danger-color);
            }
            
            .alertv2-error .alertv2-icon {
                color: var(--danger-color);
            }
            
            .alertv2-warning {
                border-left: 5px solid var(--warning-color);
            }
            
            .alertv2-warning .alertv2-icon {
                color: var(--warning-color);
            }
            
            .alertv2-info {
                border-left: 5px solid var(--info-color);
            }
            
            .alertv2-info .alertv2-icon {
                color: var(--info-color);
            }
            
            .alertv2-content {
                flex: 1;
            }
            
            .alertv2-title {
                font-weight: bold;
                margin-bottom: 5px;
            }
            
            .alertv2-message {
                margin-bottom: 0;
            }
            
            .alertv2-close {
                background: none;
                border: none;
                font-size: 18px;
                cursor: pointer;
                color: #888;
                padding: 0 5px;
            }
            
            .alertv2-close:hover {
                color: #333;
            }
        `;
        document.head.appendChild(style);
        
        // Konteyner oluştur
        const container = document.createElement('div');
        container.className = 'alertv2-container';
        document.body.appendChild(container);
        
        this.container = container;
    },
    
    create: function(options) {
        if (!this.container) {
            this.init();
        }
        
        const defaults = {
            type: 'info',
            title: '',
            message: '',
            duration: 5000,
            closeButton: true,
            callback: null
        };
        
        const config = { ...defaults, ...options };
        
        // İkon belirle
        let icon = '';
        switch (config.type) {
            case 'success':
                icon = '✓';
                break;
            case 'error':
                icon = '✗';
                break;
            case 'warning':
                icon = '⚠';
                break;
            case 'info':
            default:
                icon = 'ℹ';
                break;
        }
        
        // Uyarı elementini oluştur
        const alert = document.createElement('div');
        alert.className = `alertv2 alertv2-${config.type}`;
        
        alert.innerHTML = `
            <div class="alertv2-icon">${icon}</div>
            <div class="alertv2-content">
                ${config.title ? `<div class="alertv2-title">${config.title}</div>` : ''}
                <p class="alertv2-message">${config.message}</p>
            </div>
            ${config.closeButton ? `<button class="alertv2-close">&times;</button>` : ''}
        `;
        
        // Konteynere ekle
        this.container.appendChild(alert);
        
        // Animasyon için gecikme
        setTimeout(() => {
            alert.classList.add('show');
        }, 10);
        
        // Kapatma işlevi
        const close = () => {
            alert.classList.remove('show');
            
            setTimeout(() => {
                alert.remove();
                
                if (typeof config.callback === 'function') {
                    config.callback();
                }
            }, 300);
        };
        
        // Kapatma düğmesi için olay dinleyicisi
        if (config.closeButton) {
            const closeBtn = alert.querySelector('.alertv2-close');
            closeBtn.addEventListener('click', close);
        }
        
        // Otomatik kapanma
        if (config.duration > 0) {
            setTimeout(close, config.duration);
        }
        
        return {
            close: close
        };
    },
    
    success: function(message, title = 'Başarılı', options = {}) {
        return this.create({
            type: 'success',
            title: title,
            message: message,
            ...options
        });
    },
    
    error: function(message, title = 'Hata', options = {}) {
        return this.create({
            type: 'error',
            title: title,
            message: message,
            ...options
        });
    },
    
    warning: function(message, title = 'Uyarı', options = {}) {
        return this.create({
            type: 'warning',
            title: title,
            message: message,
            ...options
        });
    },
    
    info: function(message, title = 'Bilgi', options = {}) {
        return this.create({
            type: 'info',
            title: title,
            message: message,
            ...options
        });
    }
}; 