<?php
/**
 * Şiir Alıntıları Sistemi - Yardımcı Fonksiyonlar
 */

// Oturum başlatma - sadece oturum başlatılmamışsa çalışır
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Temizleme fonksiyonu - XSS koruması
function temizle($veri) {
    $veri = trim($veri);
    $veri = stripslashes($veri);
    $veri = htmlspecialchars($veri, ENT_QUOTES, 'UTF-8');
    return $veri;
}

// Seflink (URL için string düzenleme) fonksiyonu
function seflink($str) {
    $str = mb_strtolower($str, 'UTF-8');
    $str = str_replace(
        ['ı', 'ğ', 'ü', 'ş', 'ö', 'ç', 'İ', 'Ğ', 'Ü', 'Ş', 'Ö', 'Ç'],
        ['i', 'g', 'u', 's', 'o', 'c', 'i', 'g', 'u', 's', 'o', 'c'],
        $str
    );
    $str = preg_replace('/[^a-z0-9]/', '-', $str);
    $str = preg_replace('/-+/', '-', $str);
    $str = trim($str, '-');
    return $str;
}

// Tarih formatlama fonksiyonu
function tarihFormat($tarih) {
    $zaman = strtotime($tarih);
    return date('d.m.Y H:i', $zaman);
}

// Kısa metin oluşturma fonksiyonu
function kisaMetin($metin, $uzunluk = 150) {
    if (strlen($metin) > $uzunluk) {
        $metin = mb_substr($metin, 0, $uzunluk, 'UTF-8') . '...';
    }
    return $metin;
}

// Uyarı mesajı oluşturma fonksiyonu
function uyariOlustur($mesaj, $tur = 'success') {
    $_SESSION['uyari'] = [
        'mesaj' => $mesaj,
        'tur' => $tur
    ];
}

// Uyarı gösterme fonksiyonu
function uyariGoster() {
    if (isset($_SESSION['uyari'])) {
        $mesaj = $_SESSION['uyari']['mesaj'];
        $tur = $_SESSION['uyari']['tur'];
        
        echo '<div class="alert alert-' . $tur . '">';
        echo '<span class="close-alert">&times;</span>';
        echo $mesaj;
        echo '</div>';
        
        unset($_SESSION['uyari']);
    }
}

// Oturum kontrolü
function oturumKontrol() {
    if (!isset($_SESSION['kullanici_id'])) {
        uyariOlustur('Bu sayfaya erişim için giriş yapmalısınız!', 'danger');
        header('Location: giris.php');
        exit;
    }
}

/**
 * Admin kontrolü
 * 
 * @return void
 */
function adminKontrol() {
    // Session başlat
    if (session_status() == PHP_SESSION_NONE) {
        session_start();
    }
    
    // Admin yetkisi kontrolü
    if (!isset($_SESSION['kullanici_id']) || !isset($_SESSION['kullanici_yetki']) || $_SESSION['kullanici_yetki'] != 'admin') {
        // Admin dizininden mi kontrol et
        $current_path = $_SERVER['PHP_SELF'];
        if (strpos($current_path, '/admin/') !== false) {
            header('Location: ../giris.php');
        } else {
            header('Location: giris.php');
        }
        exit;
    }
}

// Alıntı getirme fonksiyonu
function alinitilariGetir($db, $limit = 10, $sayfa = 1, $kategori = null) {
    $baslangic = ($sayfa - 1) * $limit;
    
    $params = [];
    $sql = "SELECT y.*, GROUP_CONCAT(k.kategori_adi) as kategoriler 
            FROM yazilar y 
            LEFT JOIN yazi_kategori yk ON y.id = yk.yazi_id 
            LEFT JOIN kategoriler k ON yk.kategori_id = k.id 
            WHERE y.durum = 1";
    
    if ($kategori) {
        $sql .= " AND yk.kategori_id = :kategori_id";
        $params[':kategori_id'] = $kategori;
    }
    
    $sql .= " GROUP BY y.id 
              ORDER BY y.eklenme_tarihi DESC 
              LIMIT :baslangic, :limit";
    
    $stmt = $db->prepare($sql);
    $stmt->bindValue(':baslangic', $baslangic, PDO::PARAM_INT);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    
    if ($kategori) {
        $stmt->bindValue(':kategori_id', $kategori, PDO::PARAM_INT);
    }
    
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Alıntı detayı getirme fonksiyonu
function alintiDetay($db, $id) {
    $sql = "SELECT y.*, GROUP_CONCAT(k.kategori_adi) as kategoriler 
            FROM yazilar y 
            LEFT JOIN yazi_kategori yk ON y.id = yk.yazi_id 
            LEFT JOIN kategoriler k ON yk.kategori_id = k.id 
            WHERE y.id = :id AND y.durum = 1 
            GROUP BY y.id";
    
    $stmt = $db->prepare($sql);
    $stmt->bindValue(':id', $id, PDO::PARAM_INT);
    $stmt->execute();
    
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

// Toplam yazı sayısını getirme fonksiyonu
function toplamYaziSayisi($db, $kategori = null) {
    $sql = "SELECT COUNT(*) as toplam FROM yazilar WHERE durum = 1";
    $params = [];
    
    if ($kategori) {
        $sql = "SELECT COUNT(*) as toplam 
                FROM yazilar y 
                INNER JOIN yazi_kategori yk ON y.id = yk.yazi_id 
                WHERE y.durum = 1 AND yk.kategori_id = :kategori_id";
        $params[':kategori_id'] = $kategori;
    }
    
    $stmt = $db->prepare($sql);
    
    if ($kategori) {
        $stmt->bindValue(':kategori_id', $kategori, PDO::PARAM_INT);
    }
    
    $stmt->execute();
    $sonuc = $stmt->fetch(PDO::FETCH_ASSOC);
    
    return $sonuc['toplam'];
}

// Kategorileri getirme fonksiyonu
function kategorileriGetir($db) {
    $sql = "SELECT k.*, COUNT(yk.yazi_id) as yazi_sayisi 
            FROM kategoriler k
            LEFT JOIN yazi_kategori yk ON k.id = yk.kategori_id
            GROUP BY k.id
            ORDER BY k.kategori_adi";
    
    $stmt = $db->prepare($sql);
    $stmt->execute();
    
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Son alıntıları getirme fonksiyonu
function sonAlintilariGetir($db, $limit = 5) {
    $sql = "SELECT * FROM yazilar WHERE durum = 1 ORDER BY eklenme_tarihi DESC LIMIT :limit";
    
    $stmt = $db->prepare($sql);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->execute();
    
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Şifre haşleme fonksiyonu
function sifreHashle($sifre) {
    return password_hash($sifre, PASSWORD_BCRYPT);
}

// Şifre doğrulama fonksiyonu
function sifreDogrula($girilen_sifre, $hash) {
    return password_verify($girilen_sifre, $hash);
}

// Sayfalama fonksiyonu
function sayfalama($toplamSayfa, $aktifSayfa, $url) {
    if ($toplamSayfa <= 1) {
        return '';
    }
    
    $html = '<ul class="pagination">';
    
    // Önceki sayfa
    if ($aktifSayfa > 1) {
        $html .= '<li><a href="' . $url . '?sayfa=' . ($aktifSayfa - 1) . '">&laquo;</a></li>';
    } else {
        $html .= '<li class="disabled"><a>&laquo;</a></li>';
    }
    
    // Sayfa numaraları
    $baslangic = max(1, $aktifSayfa - 2);
    $bitis = min($toplamSayfa, $aktifSayfa + 2);
    
    if ($baslangic > 1) {
        $html .= '<li><a href="' . $url . '?sayfa=1">1</a></li>';
        if ($baslangic > 2) {
            $html .= '<li class="disabled"><a>...</a></li>';
        }
    }
    
    for ($i = $baslangic; $i <= $bitis; $i++) {
        if ($i == $aktifSayfa) {
            $html .= '<li class="active"><a>' . $i . '</a></li>';
        } else {
            $html .= '<li><a href="' . $url . '?sayfa=' . $i . '">' . $i . '</a></li>';
        }
    }
    
    if ($bitis < $toplamSayfa) {
        if ($bitis < $toplamSayfa - 1) {
            $html .= '<li class="disabled"><a>...</a></li>';
        }
        $html .= '<li><a href="' . $url . '?sayfa=' . $toplamSayfa . '">' . $toplamSayfa . '</a></li>';
    }
    
    // Sonraki sayfa
    if ($aktifSayfa < $toplamSayfa) {
        $html .= '<li><a href="' . $url . '?sayfa=' . ($aktifSayfa + 1) . '">&raquo;</a></li>';
    } else {
        $html .= '<li class="disabled"><a>&raquo;</a></li>';
    }
    
    $html .= '</ul>';
    
    return $html;
}

/**
 * Bir oturum mesajı ayarlar
 * @param string $type Mesaj tipi (success, error, warning, info)
 * @param string $message Mesaj içeriği
 * @return void
 */
function setSessionMessage($type, $message) {
    $_SESSION['messages'][] = [
        'type' => $type,
        'text' => $message
    ];
}

/**
 * Oturum mesajlarını alır ve temizler
 * @return array Mesajlar
 */
function getSessionMessages() {
    $messages = [];
    if (isset($_SESSION['messages'])) {
        $messages = $_SESSION['messages'];
        unset($_SESSION['messages']);
    }
    return $messages;
}

/**
 * Oturum mesajlarını HTML olarak görüntüler
 * @return string Mesaj HTML
 */
function displaySessionMessages() {
    $messages = getSessionMessages();
    $html = '';
    
    if (!empty($messages)) {
        foreach ($messages as $message) {
            $html .= '<div class="alert alert-' . $message['type'] . '">' . $message['text'] . '</div>';
        }
    }
    
    return $html;
}
?> 