<?php
// Gerekli dosyaları include et
require_once '../inc/autoload.php';

// Admin sınıfını başlat
$admin = new Admin($conn);

// Admin oturum kontrolü
if (!$admin->isLoggedIn()) {
    header('Location: login.php');
    exit;
}

// Site ayarlarını al
$settings = getSiteSettings($conn);

// Sayfa başlığını ayarla
$pageTitle = getPageTitle('Site Ayarları', $conn);

// Aktif sayfa
$active_page = 'settings';

// Hata ve başarı mesajları
$message = '';
$messageType = '';

// CSRF token oluştur veya kontrol et
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf_token = $_SESSION['csrf_token'];

// Ayarları güncelleme işlemi
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // CSRF kontrolü
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $message = 'Güvenlik doğrulaması başarısız oldu. Lütfen tekrar deneyin.';
        $messageType = 'danger';
    } else {
        // Girişleri temizle ve doğrula
        $site_title = filter_input(INPUT_POST, 'site_title', FILTER_SANITIZE_STRING);
        $site_description = filter_input(INPUT_POST, 'site_description', FILTER_SANITIZE_STRING);
        $site_keywords = filter_input(INPUT_POST, 'site_keywords', FILTER_SANITIZE_STRING);
        
        // İletişim bilgilerini temizle
        $contact_email = filter_input(INPUT_POST, 'contact_email', FILTER_SANITIZE_EMAIL);
        $contact_phone = filter_input(INPUT_POST, 'contact_phone', FILTER_SANITIZE_STRING);
        $contact_address = filter_input(INPUT_POST, 'contact_address', FILTER_SANITIZE_STRING);
        $contact_website = filter_input(INPUT_POST, 'contact_website', FILTER_SANITIZE_URL);
        
        // HTML kodlarını izin verilen etiketlere göre filtrele
        $adsense_code = htmlspecialchars($_POST['adsense_code'] ?? '', ENT_QUOTES, 'UTF-8');
        $analytics_code = htmlspecialchars($_POST['analytics_code'] ?? '', ENT_QUOTES, 'UTF-8');
        
        // Doğrulamalar
        if (empty($site_title)) {
            $message = 'Site başlığı gereklidir!';
            $messageType = 'danger';
        } elseif (strlen($site_title) > 100) {
            $message = 'Site başlığı 100 karakterden uzun olamaz!';
            $messageType = 'danger';
        } elseif ($contact_email && !filter_var($contact_email, FILTER_VALIDATE_EMAIL)) {
            $message = 'Geçerli bir e-posta adresi girin!';
            $messageType = 'danger';
        } else {
            $data = array(
                'site_title' => $site_title,
                'site_description' => $site_description,
                'site_keywords' => $site_keywords,
                'adsense_code' => $adsense_code,
                'analytics_code' => $analytics_code,
                'contact_email' => $contact_email,
                'contact_phone' => $contact_phone,
                'contact_address' => $contact_address,
                'contact_website' => $contact_website
            );
            
            $result = $admin->updateSettings($data);
            
            if ($result['success']) {
                $message = $result['message'];
                $messageType = 'success';
                
                // CSRF token yenile
                $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
                $csrf_token = $_SESSION['csrf_token'];
                
                // Güncellenmiş ayarları al
                $settings = getSiteSettings($conn);
            } else {
                $message = $result['message'];
                $messageType = 'danger';
            }
        }
    }
}

// Header'ı dahil et
include 'includes/header.php';
?>

<!-- Settings İçeriği -->
<div class="row mb-4">
    <div class="col-12">
        <h2 class="mb-4"><i class="fas fa-cog me-2"></i>Site Ayarları</h2>
    </div>
</div>

<div class="row">
    <div class="col-md-10 col-lg-8">
        <div class="card shadow-sm">
            <div class="card-header bg-white">
                <ul class="nav nav-tabs card-header-tabs" id="settingsTabs" role="tablist">
                    <li class="nav-item" role="presentation">
                        <button class="nav-link active" id="general-tab" data-bs-toggle="tab" data-bs-target="#general" type="button" role="tab" aria-controls="general" aria-selected="true">
                            <i class="fas fa-globe me-1"></i> Genel Ayarlar
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="contact-tab" data-bs-toggle="tab" data-bs-target="#contact" type="button" role="tab" aria-controls="contact" aria-selected="false">
                            <i class="fas fa-envelope me-1"></i> İletişim Bilgileri
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="google-tab" data-bs-toggle="tab" data-bs-target="#google" type="button" role="tab" aria-controls="google" aria-selected="false">
                            <i class="fab fa-google me-1"></i> Google Ayarları
                        </button>
                    </li>
                </ul>
            </div>
            <div class="card-body">
                <form action="settings.php" method="post" autocomplete="off" novalidate>
                    <!-- CSRF Token -->
                    <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                    
                    <div class="tab-content" id="settingsTabsContent">
                        <!-- Genel Ayarlar Sekmesi -->
                        <div class="tab-pane fade show active" id="general" role="tabpanel" aria-labelledby="general-tab">
                            <h5 class="border-bottom pb-2 mb-4">Genel Site Ayarları</h5>
                            
                            <!-- Site Başlığı -->
                            <div class="mb-3">
                                <label for="site_title" class="form-label">Site Başlığı</label>
                                <input type="text" class="form-control" id="site_title" name="site_title" value="<?php echo sanitize($settings['site_title']); ?>" required maxlength="100">
                                <div class="form-text">Sitenizin başlığı, tarayıcı sekmesinde ve site başlığında görünür. (Max: 100 karakter)</div>
                            </div>
                            
                            <!-- Site Açıklaması -->
                            <div class="mb-3">
                                <label for="site_description" class="form-label">Site Açıklaması</label>
                                <textarea class="form-control" id="site_description" name="site_description" rows="3" maxlength="255"><?php echo sanitize($settings['site_description']); ?></textarea>
                                <div class="form-text">Sitenizin kısa açıklaması, meta açıklamasında ve arama sonuçlarında görünür. (Max: 255 karakter)</div>
                            </div>
                            
                            <!-- Anahtar Kelimeler -->
                            <div class="mb-3">
                                <label for="site_keywords" class="form-label">Anahtar Kelimeler</label>
                                <textarea class="form-control" id="site_keywords" name="site_keywords" rows="3" maxlength="255"><?php echo sanitize($settings['site_keywords']); ?></textarea>
                                <div class="form-text">SEO için anahtar kelimeler, virgülle ayırın. (Max: 255 karakter)</div>
                            </div>
                        </div>
                        
                        <!-- İletişim Bilgileri Sekmesi -->
                        <div class="tab-pane fade" id="contact" role="tabpanel" aria-labelledby="contact-tab">
                            <h5 class="border-bottom pb-2 mb-4">İletişim Bilgileri</h5>
                            
                            <!-- E-posta Adresi -->
                            <div class="mb-3">
                                <label for="contact_email" class="form-label">E-posta Adresi</label>
                                <input type="email" class="form-control" id="contact_email" name="contact_email" value="<?php echo sanitize($settings['contact_email']); ?>" maxlength="100">
                                <div class="form-text">Site footer'ında görüntülenecek e-posta adresi.</div>
                            </div>
                            
                            <!-- Telefon Numarası -->
                            <div class="mb-3">
                                <label for="contact_phone" class="form-label">Telefon Numarası</label>
                                <input type="text" class="form-control" id="contact_phone" name="contact_phone" value="<?php echo sanitize($settings['contact_phone']); ?>" maxlength="20" pattern="[0-9+\-\s()]*">
                                <div class="form-text">İletişim için telefon numarası (opsiyonel). Sadece rakam, +, -, boşluk ve parantez karakterleri kullanabilirsiniz.</div>
                            </div>
                            
                            <!-- Adres -->
                            <div class="mb-3">
                                <label for="contact_address" class="form-label">Adres</label>
                                <textarea class="form-control" id="contact_address" name="contact_address" rows="3" maxlength="255"><?php echo sanitize($settings['contact_address']); ?></textarea>
                                <div class="form-text">Fiziksel adres bilgisi (opsiyonel). (Max: 255 karakter)</div>
                            </div>
                            
                            <!-- Web Sitesi -->
                            <div class="mb-3">
                                <label for="contact_website" class="form-label">Web Sitesi</label>
                                <input type="url" class="form-control" id="contact_website" name="contact_website" value="<?php echo sanitize($settings['contact_website']); ?>" maxlength="100">
                                <div class="form-text">Site footer'ında görüntülenecek web sitesi adresi. (Örn: https://www.example.com)</div>
                            </div>
                        </div>
                        
                        <!-- Google Ayarları Sekmesi -->
                        <div class="tab-pane fade" id="google" role="tabpanel" aria-labelledby="google-tab">
                            <h5 class="border-bottom pb-2 mb-4">Google Entegrasyonları</h5>
                            
                            <!-- Google AdSense Kodu -->
                            <div class="mb-4">
                                <label for="adsense_code" class="form-label">Google AdSense Kodu</label>
                                <textarea class="form-control" id="adsense_code" name="adsense_code" rows="5"><?php echo htmlspecialchars($settings['adsense_code']); ?></textarea>
                                <div class="form-text">Google AdSense kodunuzu buraya yapıştırın. Bu kod site genelinde reklam alanlarında kullanılacaktır. <strong>Not:</strong> Sadece izin verilen Google kodları çalışacaktır.</div>
                            </div>
                            
                            <!-- Google Analytics Kodu -->
                            <div class="mb-3">
                                <label for="analytics_code" class="form-label">Google Analytics Kodu</label>
                                <textarea class="form-control" id="analytics_code" name="analytics_code" rows="5"><?php echo htmlspecialchars($settings['analytics_code']); ?></textarea>
                                <div class="form-text">Google Analytics kodunuzu buraya yapıştırın. Bu kod site trafiğini izlemek için kullanılacaktır. <strong>Not:</strong> Sadece izin verilen Google kodları çalışacaktır.</div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Kaydet Butonu -->
                    <div class="d-grid gap-2 d-md-flex justify-content-md-end mt-4">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-1"></i> Tüm Ayarları Kaydet
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-md-10 col-lg-4 mt-4 mt-lg-0">
        <div class="card shadow-sm bg-light">
            <div class="card-header bg-white">
                <h5 class="mb-0">Yardım & Bilgi</h5>
            </div>
            <div class="card-body">
                <div class="mb-4">
                    <h6><i class="fas fa-globe me-2"></i>Genel Ayarlar</h6>
                    <p class="small">Site başlığı, açıklaması ve anahtar kelimeler, sitenizin arama motorlarında daha iyi sıralanmasına yardımcı olur. Bu alanları doğru içerikle doldurmaya özen gösterin.</p>
                </div>
                
                <div class="mb-4">
                    <h6><i class="fas fa-envelope me-2"></i>İletişim Bilgileri</h6>
                    <p class="small">Footer bölümünde görüntülenecek iletişim bilgilerini düzenleyin. Bu bilgiler ziyaretçilerin size ulaşmasını sağlar.</p>
                </div>
                
                <div class="mb-4">
                    <h6><i class="fas fa-ad me-2"></i>Google AdSense</h6>
                    <p class="small">AdSense kodunuzu buraya eklediğinizde, site içerisinde belirlenen reklam alanlarında reklamlar otomatik olarak görüntülenecektir.</p>
                </div>
                
                <div class="mb-4">
                    <h6><i class="fas fa-chart-line me-2"></i>Google Analytics</h6>
                    <p class="small">Analytics kodunu ekleyerek sitenizin trafik analizlerini takip edebilirsiniz. Kodunuzu buraya eklediğinizde, tüm sayfalarda otomatik olarak çalışacaktır.</p>
                </div>
                
                <div>
                    <h6><i class="fas fa-shield-alt me-2"></i>Güvenlik İpuçları</h6>
                    <p class="small">
                        <ul class="small ps-3">
                            <li>Özellikle AdSense ve Analytics gibi script alanlarına sadece resmi Google kodlarını ekleyin.</li>
                            <li>Düzenli olarak admin şifrenizi değiştirin.</li>
                            <li>Her zaman güncel ve güvenli bir tarayıcı kullanın.</li>
                        </ul>
                    </p>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Tab hatırlama için JavaScript -->
<script>
    // Sayfa yüklendiğinde son seçilen sekmeyi hatırla
    document.addEventListener('DOMContentLoaded', function() {
        // LocalStorage'dan son seçilen sekmeyi al
        const lastTab = localStorage.getItem('selectedSettingsTab');
        
        // Eğer kayıtlı bir sekme varsa, onu aktif et
        if (lastTab) {
            const tabEl = document.querySelector(`#settingsTabs button[data-bs-target="${lastTab}"]`);
            if (tabEl) {
                const tab = new bootstrap.Tab(tabEl);
                tab.show();
            }
        }
        
        // Sekme değişikliklerini dinle ve seçilen sekmeyi kaydet
        const tabEls = document.querySelectorAll('#settingsTabs button');
        tabEls.forEach(tabEl => {
            tabEl.addEventListener('shown.bs.tab', function (e) {
                // Seçilen sekmenin ID'sini kaydet
                localStorage.setItem('selectedSettingsTab', e.target.getAttribute('data-bs-target'));
            });
        });
        
        // Form doğrulama
        const form = document.querySelector('form');
        form.addEventListener('submit', function(event) {
            // Site başlığı kontrolü
            const siteTitleInput = document.getElementById('site_title');
            if (!siteTitleInput.value.trim()) {
                siteTitleInput.classList.add('is-invalid');
                event.preventDefault();
                return false;
            }
            
            // E-posta doğrulama
            const emailInput = document.getElementById('contact_email');
            if (emailInput.value && !validateEmail(emailInput.value)) {
                emailInput.classList.add('is-invalid');
                event.preventDefault();
                return false;
            }
        });
        
        // Input alanlarına validation eventleri ekle
        document.querySelectorAll('input, textarea').forEach(input => {
            input.addEventListener('input', function() {
                this.classList.remove('is-invalid');
            });
        });
        
        // Email doğrulama fonksiyonu
        function validateEmail(email) {
            const re = /^(([^<>()\[\]\\.,;:\s@"]+(\.[^<>()\[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
            return re.test(String(email).toLowerCase());
        }
    });
</script>

<?php
// Footer'ı dahil et
include 'includes/footer.php';
?> 