<?php

class Admin {
    private $conn;
    
    public function __construct($conn) {
        $this->conn = $conn;
    }
    
    // Admin oturum kontrolü
    public function isLoggedIn() {
        return isset($_SESSION['admin_id']);
    }
    
    // Admin girişi
    public function login($username, $password) {
        // SQL sorgusu
        $sql = "SELECT id, username, password FROM admin WHERE username = ?";
        $stmt = mysqli_prepare($this->conn, $sql);
        mysqli_stmt_bind_param($stmt, "s", $username);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        
        if (mysqli_num_rows($result) == 1) {
            $row = mysqli_fetch_assoc($result);
            
            // Şifre kontrolü
            if (password_verify($password, $row['password'])) {
                // Oturum başlatma
                $_SESSION['admin_id'] = $row['id'];
                $_SESSION['admin_username'] = $row['username'];
                
                return array("success" => true, "message" => "Giriş başarılı.");
            } else {
                return array("success" => false, "message" => "Geçersiz şifre.");
            }
        } else {
            return array("success" => false, "message" => "Geçersiz kullanıcı adı.");
        }
    }
    
    // Admin çıkış
    public function logout() {
        // Admin oturum değişkenlerini temizle
        unset($_SESSION['admin_id']);
        unset($_SESSION['admin_username']);
        
        return true;
    }
    
    // Kullanıcıları listele
    public function listUsers() {
        $sql = "SELECT id, username, email, created_at FROM users ORDER BY created_at DESC";
        $result = mysqli_query($this->conn, $sql);
        
        $users = array();
        while ($row = mysqli_fetch_assoc($result)) {
            $users[] = $row;
        }
        
        return $users;
    }
    
    // Fotoğrafları listele
    public function listPhotos() {
        $sql = "SELECT p.*, u.username FROM photos p LEFT JOIN users u ON p.user_id = u.id ORDER BY p.created_at DESC";
        $result = mysqli_query($this->conn, $sql);
        
        $photos = array();
        while ($row = mysqli_fetch_assoc($result)) {
            $photos[] = $row;
        }
        
        return $photos;
    }
    
    // Fotoğrafı sil
    public function deletePhoto($photoId) {
        // Fotoğraf bilgilerini al
        $sql = "SELECT file_name FROM photos WHERE id = ?";
        $stmt = mysqli_prepare($this->conn, $sql);
        mysqli_stmt_bind_param($stmt, "i", $photoId);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        
        if (mysqli_num_rows($result) > 0) {
            $row = mysqli_fetch_assoc($result);
            $filePath = "../uploads/" . $row['file_name'];
            
            // Dosyayı sil
            if (file_exists($filePath)) {
                unlink($filePath);
            }
            
            // Veritabanından sil
            $sql = "DELETE FROM photos WHERE id = ?";
            $stmt = mysqli_prepare($this->conn, $sql);
            mysqli_stmt_bind_param($stmt, "i", $photoId);
            
            if (mysqli_stmt_execute($stmt)) {
                return array("success" => true, "message" => "Fotoğraf başarıyla silindi.");
            } else {
                return array("success" => false, "message" => "Fotoğraf silinirken bir hata oluştu: " . mysqli_error($this->conn));
            }
        } else {
            return array("success" => false, "message" => "Fotoğraf bulunamadı.");
        }
    }
    
    // Kullanıcıyı sil
    public function deleteUser($userId) {
        // Kullanıcıya ait fotoğrafları silme
        $sql = "SELECT file_name FROM photos WHERE user_id = ?";
        $stmt = mysqli_prepare($this->conn, $sql);
        mysqli_stmt_bind_param($stmt, "i", $userId);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        
        while ($row = mysqli_fetch_assoc($result)) {
            $filePath = "../uploads/" . $row['file_name'];
            if (file_exists($filePath)) {
                unlink($filePath);
            }
        }
        
        // Kullanıcıya ait fotoğraf kayıtlarını silme
        $sql = "DELETE FROM photos WHERE user_id = ?";
        $stmt = mysqli_prepare($this->conn, $sql);
        mysqli_stmt_bind_param($stmt, "i", $userId);
        mysqli_stmt_execute($stmt);
        
        // Kullanıcıyı silme
        $sql = "DELETE FROM users WHERE id = ?";
        $stmt = mysqli_prepare($this->conn, $sql);
        mysqli_stmt_bind_param($stmt, "i", $userId);
        
        if (mysqli_stmt_execute($stmt)) {
            return array("success" => true, "message" => "Kullanıcı başarıyla silindi.");
        } else {
            return array("success" => false, "message" => "Kullanıcı silinirken bir hata oluştu: " . mysqli_error($this->conn));
        }
    }
    
    // Site ayarlarını güncelle
    public function updateSettings($data) {
        // Verileri temizle ve doğrula
        $site_title = $this->sanitizeInput($data['site_title']);
        $site_description = $this->sanitizeInput($data['site_description']);
        $site_keywords = $this->sanitizeInput($data['site_keywords']);
        
        // Script alanlarını ayrıca işle
        $adsense_code = htmlspecialchars($data['adsense_code'], ENT_QUOTES, 'UTF-8');
        $analytics_code = htmlspecialchars($data['analytics_code'], ENT_QUOTES, 'UTF-8');
        
        // İletişim alanlarını temizle
        $contact_email = filter_var($data['contact_email'], FILTER_SANITIZE_EMAIL);
        $contact_phone = $this->sanitizeInput($data['contact_phone']);
        $contact_address = $this->sanitizeInput($data['contact_address']);
        $contact_website = filter_var($data['contact_website'], FILTER_SANITIZE_URL);
        
        // Validation
        if (empty($site_title)) {
            return array("success" => false, "message" => "Site başlığı boş olamaz.");
        }
        
        if ($contact_email && !filter_var($contact_email, FILTER_VALIDATE_EMAIL)) {
            return array("success" => false, "message" => "Geçerli bir e-posta adresi girin.");
        }
        
        // Prepared statement kullan
        $sql = "UPDATE settings SET 
                site_title = ?, 
                site_description = ?, 
                site_keywords = ?, 
                adsense_code = ?, 
                analytics_code = ?,
                contact_email = ?,
                contact_phone = ?,
                contact_address = ?,
                contact_website = ?
                WHERE id = 1";
                
        $stmt = mysqli_prepare($this->conn, $sql);
        
        if (!$stmt) {
            return array("success" => false, "message" => "SQL sorgusu hazırlanırken bir hata oluştu: " . mysqli_error($this->conn));
        }
        
        mysqli_stmt_bind_param($stmt, "sssssssss", 
            $site_title, 
            $site_description, 
            $site_keywords, 
            $adsense_code, 
            $analytics_code, 
            $contact_email, 
            $contact_phone, 
            $contact_address, 
            $contact_website
        );
        
        if (mysqli_stmt_execute($stmt)) {
            mysqli_stmt_close($stmt);
            // Güvenlik için başarılı mesajını genel tut
            return array("success" => true, "message" => "Ayarlar başarıyla güncellendi.");
        } else {
            $error = mysqli_stmt_error($stmt);
            mysqli_stmt_close($stmt);
            // Hatayı logla ama kullanıcıya detaylı gösterme
            error_log("Settings update error: " . $error);
            return array("success" => false, "message" => "Ayarlar güncellenirken bir hata oluştu.");
        }
    }
    
    // Metin girdilerini temizleme yardımcı metodu
    private function sanitizeInput($input) {
        // Boşlukları temizle
        $input = trim($input);
        // HTML taglarını temizle
        $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
        return $input;
    }
    
    // Admin şifresini değiştir
    public function changePassword($currentPassword, $newPassword, $confirmPassword) {
        // Giriş yapmış bir admin var mı kontrol et
        if (!isset($_SESSION['admin_id'])) {
            return array("success" => false, "message" => "Oturum açılmamış.");
        }
        
        $adminId = $_SESSION['admin_id'];
        
        // Şifre kontrolü
        if (empty($currentPassword) || empty($newPassword) || empty($confirmPassword)) {
            return array("success" => false, "message" => "Tüm alanlar doldurulmalıdır.");
        }
        
        // Yeni şifre ve onay şifresi eşleşiyor mu?
        if ($newPassword !== $confirmPassword) {
            return array("success" => false, "message" => "Yeni şifre ve onay şifresi eşleşmiyor.");
        }
        
        // Şifre gücü kontrolü
        if (strlen($newPassword) < 8) {
            return array("success" => false, "message" => "Şifre en az 8 karakter uzunluğunda olmalıdır.");
        }
        
        // Mevcut şifreyi doğrula
        $sql = "SELECT password FROM admin WHERE id = ?";
        $stmt = mysqli_prepare($this->conn, $sql);
        mysqli_stmt_bind_param($stmt, "i", $adminId);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        
        if (mysqli_num_rows($result) == 1) {
            $row = mysqli_fetch_assoc($result);
            
            // Mevcut şifre kontrolü
            if (!password_verify($currentPassword, $row['password'])) {
                return array("success" => false, "message" => "Mevcut şifre yanlış.");
            }
            
            // Yeni şifreyi hashle
            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
            
            // Şifreyi güncelle
            $updateSql = "UPDATE admin SET password = ? WHERE id = ?";
            $updateStmt = mysqli_prepare($this->conn, $updateSql);
            mysqli_stmt_bind_param($updateStmt, "si", $hashedPassword, $adminId);
            
            if (mysqli_stmt_execute($updateStmt)) {
                mysqli_stmt_close($updateStmt);
                return array("success" => true, "message" => "Şifre başarıyla değiştirildi.");
            } else {
                $error = mysqli_stmt_error($updateStmt);
                mysqli_stmt_close($updateStmt);
                error_log("Password change error: " . $error);
                return array("success" => false, "message" => "Şifre değiştirilirken bir hata oluştu.");
            }
        } else {
            return array("success" => false, "message" => "Admin hesabı bulunamadı.");
        }
    }
}