<?php

class User {
    private $conn;
    
    public function __construct($conn) {
        $this->conn = $conn;
    }
    
    // Kullanıcı kaydı
    public function register($username, $email, $password) {
        // Kullanıcı adı ve e-posta kontrolü
        if ($this->isUsernameExists($username)) {
            return array("success" => false, "message" => "Bu kullanıcı adı zaten kullanılıyor.");
        }
        
        if ($this->isEmailExists($email)) {
            return array("success" => false, "message" => "Bu e-posta adresi zaten kullanılıyor.");
        }
        
        // Şifreyi hashleme
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
        
        // Kayıt tarihi
        $created_at = date('Y-m-d H:i:s');
        
        // SQL sorgusu
        $sql = "INSERT INTO users (username, email, password, created_at) VALUES (?, ?, ?, ?)";
        $stmt = mysqli_prepare($this->conn, $sql);
        mysqli_stmt_bind_param($stmt, "ssss", $username, $email, $hashed_password, $created_at);
        
        if (mysqli_stmt_execute($stmt)) {
            return array("success" => true, "message" => "Kayıt başarıyla tamamlandı. Giriş yapabilirsiniz.");
        } else {
            return array("success" => false, "message" => "Kayıt sırasında bir hata oluştu: " . mysqli_error($this->conn));
        }
    }
    
    // Kullanıcı girişi
    public function login($email, $password) {
        // SQL sorgusu
        $sql = "SELECT id, username, email, password FROM users WHERE email = ?";
        $stmt = mysqli_prepare($this->conn, $sql);
        mysqli_stmt_bind_param($stmt, "s", $email);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        
        if (mysqli_num_rows($result) == 1) {
            $row = mysqli_fetch_assoc($result);
            
            // Şifre kontrolü
            if (password_verify($password, $row['password'])) {
                // Oturum başlatma
                $_SESSION['user_id'] = $row['id'];
                $_SESSION['username'] = $row['username'];
                $_SESSION['email'] = $row['email'];
                
                return array("success" => true, "message" => "Giriş başarılı.");
            } else {
                return array("success" => false, "message" => "Geçersiz şifre.");
            }
        } else {
            return array("success" => false, "message" => "Bu e-posta adresi kayıtlı değil.");
        }
    }
    
    // Kullanıcı adı kontrolü
    private function isUsernameExists($username) {
        $sql = "SELECT id FROM users WHERE username = ?";
        $stmt = mysqli_prepare($this->conn, $sql);
        mysqli_stmt_bind_param($stmt, "s", $username);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_store_result($stmt);
        
        return mysqli_stmt_num_rows($stmt) > 0;
    }
    
    // E-posta kontrolü
    private function isEmailExists($email) {
        $sql = "SELECT id FROM users WHERE email = ?";
        $stmt = mysqli_prepare($this->conn, $sql);
        mysqli_stmt_bind_param($stmt, "s", $email);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_store_result($stmt);
        
        return mysqli_stmt_num_rows($stmt) > 0;
    }
    
    // Kullanıcının yüklediği fotoğrafları getir
    public function getUserPhotos($userId) {
        $sql = "SELECT * FROM photos WHERE user_id = ? ORDER BY created_at DESC";
        $stmt = mysqli_prepare($this->conn, $sql);
        mysqli_stmt_bind_param($stmt, "i", $userId);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        
        $photos = array();
        while ($row = mysqli_fetch_assoc($result)) {
            $photos[] = $row;
        }
        
        return $photos;
    }
    
    // Çıkış yap
    public function logout() {
        // Oturum değişkenlerini temizle
        $_SESSION = array();
        
        // Oturum çerezini yok et
        if (ini_get("session.use_cookies")) {
            $params = session_get_cookie_params();
            setcookie(session_name(), '', time() - 42000,
                $params["path"], $params["domain"],
                $params["secure"], $params["httponly"]
            );
        }
        
        // Oturumu sonlandır
        session_destroy();
        
        return true;
    }
}
?> 