/**
 * Fotoğraf Upload JavaScript İşlemleri
 */

document.addEventListener('DOMContentLoaded', function() {
    // Sürükle & Bırak İşlemleri
    setupDragAndDrop();
    
    // Yapıştır & Yükle İşlemleri
    setupPasteUpload();
});

/**
 * Sürükle & Bırak İşlemlerini Ayarla
 */
function setupDragAndDrop() {
    const dropzone = document.getElementById('dropzone');
    const dropzonePreview = document.getElementById('dropzone-preview');
    
    if (!dropzone) return;
    
    // Drag over olayı
    dropzone.addEventListener('dragover', function(e) {
        e.preventDefault();
        e.stopPropagation();
        this.classList.add('border-primary');
    });
    
    // Drag leave olayı
    dropzone.addEventListener('dragleave', function(e) {
        e.preventDefault();
        e.stopPropagation();
        this.classList.remove('border-primary');
    });
    
    // Drop olayı
    dropzone.addEventListener('drop', function(e) {
        e.preventDefault();
        e.stopPropagation();
        this.classList.remove('border-primary');
        
        const files = e.dataTransfer.files;
        if (files.length > 0) {
            handleDroppedFiles(files);
        }
    });
}

/**
 * Bırakılan Dosyaları İşle
 */
function handleDroppedFiles(files) {
    const dropzonePreview = document.getElementById('dropzone-preview');
    dropzonePreview.innerHTML = ''; // Önizlemeyi temizle
    
    // Sadece ilk dosyayı işle (çoklu dosya yükleme şimdilik desteklenmiyor)
    const file = files[0];
    
    // Dosya türü kontrolü
    if (!isValidImageFile(file)) {
        showPreviewError('Geçersiz dosya türü. Sadece JPG, JPEG, PNG ve GIF formatları kabul edilir.');
        return;
    }
    
    // Dosya boyutu kontrolü (5MB)
    if (file.size > 5 * 1024 * 1024) {
        showPreviewError('Dosya boyutu çok büyük. Maksimum 5MB izin verilir.');
        return;
    }
    
    // Ön izleme göster
    const reader = new FileReader();
    reader.onload = function(e) {
        const img = document.createElement('img');
        img.src = e.target.result;
        img.className = 'img-fluid mb-3';
        img.style.maxHeight = '300px';
        
        const previewContainer = document.createElement('div');
        
        // Dosya bilgileri
        const fileInfo = document.createElement('p');
        fileInfo.className = 'mb-3';
        fileInfo.innerHTML = `
            <strong>${file.name}</strong><br>
            <small>${formatFileType(file.type)} - ${formatFileSize(file.size)}</small>
        `;
        
        // Upload butonu
        const uploadBtn = document.createElement('button');
        uploadBtn.className = 'btn btn-primary';
        uploadBtn.innerHTML = '<i class="fas fa-upload me-1"></i> Yükle';
        uploadBtn.onclick = function() {
            uploadDroppedFile(file);
        };
        
        // Hepsini ekle
        previewContainer.appendChild(img);
        previewContainer.appendChild(fileInfo);
        previewContainer.appendChild(uploadBtn);
        
        dropzonePreview.innerHTML = '';
        dropzonePreview.appendChild(previewContainer);
    };
    
    reader.readAsDataURL(file);
}

/**
 * Sürüklenen Dosyayı Yükle
 */
function uploadDroppedFile(file) {
    const formData = new FormData();
    formData.append('file', file);
    
    // Yükleme spin göster
    const dropzonePreview = document.getElementById('dropzone-preview');
    dropzonePreview.innerHTML = `
        <div class="text-center">
            <div class="spinner-border text-primary" role="status">
                <span class="visually-hidden">Yükleniyor...</span>
            </div>
            <p class="mt-2">Fotoğraf yükleniyor, lütfen bekleyin...</p>
        </div>
    `;
    
    // AJAX isteği
    fetch('index.php', {
        method: 'POST',
        body: formData
    })
    .then(response => {
        if(response.redirected) {
            // Eğer yönlendirme varsa o URL'e git
            window.location.href = response.url;
            return;
        }
        return response.text();
    })
    .then(html => {
        // Sadece yönlendirme olmadıysa bu kısım çalışacak
        if(html) {
            // Başarı mesajını kontrol et
            if(html.includes('Fotoğraf başarıyla yüklendi')) {
                // ID'yi bulmaya çalış
                const match = html.match(/photo_id=(\d+)/);
                if(match && match[1]) {
                    window.location.href = 'upload-success.php?id=' + match[1];
                    return;
                }
            }
            
            // Başarısız olursa sayfayı yenile
            location.reload();
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showPreviewError('Yükleme sırasında bir hata oluştu. Lütfen tekrar deneyin.');
    });
}

/**
 * Yapıştır & Yükle İşlemlerini Ayarla
 */
function setupPasteUpload() {
    const pasteArea = document.getElementById('pasteArea');
    const pastePreview = document.getElementById('pastePreview');
    const pasteForm = document.getElementById('pasteForm');
    const pasteData = document.getElementById('pasteData');
    
    if (!pasteArea || !pastePreview || !pasteForm || !pasteData) return;
    
    // Tüm sayfada paste olayını dinle
    document.addEventListener('paste', function(e) {
        // Eğer aktif sekme paste sekmesi değilse işlemi yapma
        const pasteTab = document.getElementById('paste-tab');
        if (!pasteTab.classList.contains('active')) return;
        
        const items = e.clipboardData.items;
        
        // Kopyalanan verileri kontrol et
        for (let i = 0; i < items.length; i++) {
            if (items[i].type.indexOf('image') !== -1) {
                const blob = items[i].getAsFile();
                
                // Dosya boyutu kontrolü (5MB)
                if (blob.size > 5 * 1024 * 1024) {
                    showPasteError('Dosya boyutu çok büyük. Maksimum 5MB izin verilir.');
                    return;
                }
                
                // Önizleme göster
                const reader = new FileReader();
                reader.onload = function(e) {
                    // Önizleme resmi
                    const img = document.createElement('img');
                    img.src = e.target.result;
                    img.className = 'img-fluid mb-3';
                    img.style.maxHeight = '300px';
                    
                    // Dosya bilgileri
                    const fileInfo = document.createElement('p');
                    fileInfo.className = 'mb-3';
                    fileInfo.innerHTML = `
                        <strong>Yapıştırılan Görsel</strong><br>
                        <small>${formatFileType(blob.type)} - ${formatFileSize(blob.size)}</small>
                    `;
                    
                    // Önizlemeyi temizle ve yeni içerikleri ekle
                    pastePreview.innerHTML = '';
                    pastePreview.appendChild(img);
                    pastePreview.appendChild(fileInfo);
                    
                    // Form verisi güncelle ve göster
                    pasteData.value = e.target.result;
                    pasteForm.style.display = 'block';
                };
                
                reader.readAsDataURL(blob);
                break;
            }
        }
    });
    
    // Yapıştırma alanına tıklama olayı
    pasteArea.addEventListener('click', function() {
        this.classList.add('border-primary');
        setTimeout(() => {
            this.classList.remove('border-primary');
        }, 300);
    });
}

/**
 * Hata Mesajını Önizlemede Göster (Sürükle & Bırak)
 */
function showPreviewError(message) {
    const dropzonePreview = document.getElementById('dropzone-preview');
    dropzonePreview.innerHTML = `
        <div class="alert alert-danger" role="alert">
            <i class="fas fa-exclamation-circle me-2"></i> ${message}
        </div>
    `;
}

/**
 * Hata Mesajını Önizlemede Göster (Yapıştır)
 */
function showPasteError(message) {
    const pastePreview = document.getElementById('pastePreview');
    pastePreview.innerHTML = `
        <div class="alert alert-danger" role="alert">
            <i class="fas fa-exclamation-circle me-2"></i> ${message}
        </div>
    `;
}

/**
 * Dosya Türünü Formatlama
 */
function formatFileType(mimeType) {
    const types = {
        'image/jpeg': 'JPEG',
        'image/jpg': 'JPG',
        'image/png': 'PNG',
        'image/gif': 'GIF'
    };
    
    return types[mimeType] || 'Görsel';
}

/**
 * Dosya Boyutunu Formatlama
 */
function formatFileSize(bytes) {
    if (bytes >= 1073741824) {
        return (bytes / 1073741824).toFixed(2) + ' GB';
    } else if (bytes >= 1048576) {
        return (bytes / 1048576).toFixed(2) + ' MB';
    } else if (bytes >= 1024) {
        return (bytes / 1024).toFixed(2) + ' KB';
    } else if (bytes > 1) {
        return bytes + ' bytes';
    } else if (bytes == 1) {
        return bytes + ' byte';
    } else {
        return '0 bytes';
    }
}

/**
 * Dosya Uzantısını Kontrol Et
 */
function isValidImageFile(file) {
    const validTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
    return validTypes.includes(file.type);
} 